// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { serializeFactory, aadFactory, Maximum, } from '@aws-crypto/serialize';
// @ts-ignore
import { Transform as PortableTransform } from 'readable-stream';
import { needs, } from '@aws-crypto/material-management-node';
const fromUtf8 = (input) => Buffer.from(input, 'utf8');
const serialize = serializeFactory(fromUtf8);
const { finalFrameHeader, frameHeader } = serialize;
const aadUtility = aadFactory(fromUtf8);
const PortableTransformWithType = PortableTransform;
const ioTick = async () => new Promise((resolve) => setImmediate(resolve));
const noop = () => { }; // eslint-disable-line @typescript-eslint/no-empty-function
export function getFramedEncryptStream(getCipher, messageHeader, dispose, { plaintextLength, suite, }) {
    let accumulatingFrame = {
        contentLength: 0,
        content: [],
        sequenceNumber: 1,
    };
    let pathologicalDrain = noop;
    const { frameLength } = messageHeader;
    /* Precondition: plaintextLength must be within bounds.
     * The Maximum.BYTES_PER_MESSAGE is set to be within Number.MAX_SAFE_INTEGER
     * See serialize/identifiers.ts enum Maximum for more details.
     */
    needs(!plaintextLength ||
        (plaintextLength >= 0 && Maximum.BYTES_PER_MESSAGE >= plaintextLength), 'plaintextLength out of bounds.');
    /* Keeping the messageHeader, accumulatingFrame and pathologicalDrain private is the intention here.
     * It is already unlikely that these values could be touched in the current composition of streams,
     * but a different composition may change this.
     * Since we are handling the plain text here, it seems prudent to take extra measures.
     */
    return new (class FramedEncryptStream extends PortableTransformWithType {
        _transform(chunk, encoding, callback) {
            const contentLeft = frameLength - accumulatingFrame.contentLength;
            /* Precondition: Must not process more than plaintextLength.
             * The plaintextLength is the MAXIMUM value that can be encrypted.
             */
            needs(!plaintextLength || (plaintextLength -= chunk.length) >= 0, 'Encrypted data exceeded plaintextLength.');
            /* Check for early return (Postcondition): Have not accumulated a frame. */
            if (contentLeft > chunk.length) {
                // eat more
                accumulatingFrame.contentLength += chunk.length;
                accumulatingFrame.content.push(chunk);
                return callback();
            }
            accumulatingFrame.contentLength += contentLeft;
            accumulatingFrame.content.push(chunk.slice(0, contentLeft));
            // grab the tail
            const tail = chunk.slice(contentLeft);
            const encryptFrame = getEncryptFrame({
                pendingFrame: accumulatingFrame,
                messageHeader,
                getCipher,
                isFinalFrame: false,
                suite,
            });
            // Reset frame state for next frame
            const { sequenceNumber } = accumulatingFrame;
            accumulatingFrame = {
                contentLength: 0,
                content: [],
                sequenceNumber: sequenceNumber + 1,
            };
            this._flushEncryptFrame(encryptFrame)
                .then(() => this._transform(tail, encoding, callback))
                .catch(callback);
        }
        _flush(callback) {
            const encryptFrame = getEncryptFrame({
                pendingFrame: accumulatingFrame,
                messageHeader,
                getCipher,
                isFinalFrame: true,
                suite,
            });
            this._flushEncryptFrame(encryptFrame)
                .then(() => callback())
                .catch(callback);
        }
        _destroy() {
            dispose();
        }
        _read(size) {
            super._read(size);
            /* The _flushEncryptFrame encrypts and pushes the frame.
             * If this.push returns false then this stream
             * should wait until the destination stream calls read.
             * This means that _flushEncryptFrame needs to wait for some
             * indeterminate time.  I create a closure around
             * the resolution function for a promise that
             * is created in _flushEncryptFrame.  This way
             * here in _read (the implementation of read)
             * if a frame is being pushed, we can release
             * it.
             */
            pathologicalDrain();
            pathologicalDrain = noop;
        }
        async _flushEncryptFrame(encryptingFrame) {
            const { content, cipher, bodyHeader, isFinalFrame } = encryptingFrame;
            this.push(bodyHeader);
            let frameSize = 0;
            const cipherContent = [];
            for (const clearChunk of content) {
                const cipherText = cipher.update(clearChunk);
                frameSize += cipherText.length;
                cipherContent.push(cipherText);
                await ioTick();
            }
            /* Finalize the cipher and handle any tail. */
            const tail = cipher.final();
            frameSize += tail.length;
            cipherContent.push(tail);
            /* Push the authTag onto the end.  Yes, I am abusing the name. */
            cipherContent.push(cipher.getAuthTag());
            needs(frameSize === frameLength || (isFinalFrame && frameLength >= frameSize), 'Malformed frame');
            for (const cipherText of cipherContent) {
                if (!this.push(cipherText)) {
                    /* back pressure: if push returns false, wait until _read
                     * has been called.
                     */
                    await new Promise((resolve) => {
                        pathologicalDrain = resolve;
                    });
                }
            }
            if (isFinalFrame)
                this.push(null);
        }
    })();
}
export function getEncryptFrame(input) {
    const { pendingFrame, messageHeader, getCipher, isFinalFrame, suite } = input;
    const { sequenceNumber, contentLength, content } = pendingFrame;
    const { frameLength, contentType, messageId } = messageHeader;
    /* Precondition: The content length MUST correlate with the frameLength.
     * In the case of a regular frame,
     * the content length MUST strictly equal the frame length.
     * In the case of the final frame,
     * it MUST NOT be larger than the frame length.
     */
    needs(frameLength === contentLength ||
        (isFinalFrame && frameLength >= contentLength), `Malformed frame length and content length: ${JSON.stringify({
        frameLength,
        contentLength,
        isFinalFrame,
    })}`);
    const frameIv = serialize.frameIv(suite.ivLength, sequenceNumber);
    const bodyHeader = Buffer.from(isFinalFrame
        ? finalFrameHeader(sequenceNumber, frameIv, contentLength)
        : frameHeader(sequenceNumber, frameIv));
    const contentString = aadUtility.messageAADContentString({
        contentType,
        isFinalFrame,
    });
    const { buffer, byteOffset, byteLength } = aadUtility.messageAAD(messageId, contentString, sequenceNumber, contentLength);
    const cipher = getCipher(frameIv);
    cipher.setAAD(Buffer.from(buffer, byteOffset, byteLength));
    return { content, cipher, bodyHeader, isFinalFrame };
}
//# sourceMappingURL=data:application/json;base64,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