'use strict';

exports.__esModule = true;
exports.default = tokenizer;
var SINGLE_QUOTE = 39;
var DOUBLE_QUOTE = 34;
var BACKSLASH = 92;
var SLASH = 47;
var NEWLINE = 10;
var SPACE = 32;
var FEED = 12;
var TAB = 9;
var CR = 13;
var OPEN_SQUARE = 91;
var CLOSE_SQUARE = 93;
var OPEN_PARENTHESES = 40;
var CLOSE_PARENTHESES = 41;
var OPEN_CURLY = 123;
var CLOSE_CURLY = 125;
var SEMICOLON = 59;
var ASTERISK = 42;
var COLON = 58;
var AT = 64;

var RE_AT_END = /[ \n\t\r\f{}()'"\\;/[\]#]/g;
var RE_WORD_END = /[ \n\t\r\f(){}:;@!'"\\\][#]|\/(?=\*)/g;
var RE_BAD_BRACKET = /.[\\/("'\n]/;
var RE_HEX_ESCAPE = /[a-f0-9]/i;

function tokenizer(input) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

  var css = input.css.valueOf();
  var ignore = options.ignoreErrors;

  var code = void 0,
      next = void 0,
      quote = void 0,
      lines = void 0,
      last = void 0,
      content = void 0,
      escape = void 0;
  var nextLine = void 0,
      nextOffset = void 0,
      escaped = void 0,
      escapePos = void 0,
      prev = void 0,
      n = void 0,
      currentToken = void 0;

  var length = css.length;
  var offset = -1;
  var line = 1;
  var pos = 0;
  var buffer = [];
  var returned = [];

  function unclosed(what) {
    throw input.error('Unclosed ' + what, line, pos - offset);
  }

  function endOfFile() {
    return returned.length === 0 && pos >= length;
  }

  function nextToken() {
    if (returned.length) return returned.pop();
    if (pos >= length) return;

    code = css.charCodeAt(pos);
    if (code === NEWLINE || code === FEED || code === CR && css.charCodeAt(pos + 1) !== NEWLINE) {
      offset = pos;
      line += 1;
    }

    switch (code) {
      case NEWLINE:
      case SPACE:
      case TAB:
      case CR:
      case FEED:
        next = pos;
        do {
          next += 1;
          code = css.charCodeAt(next);
          if (code === NEWLINE) {
            offset = next;
            line += 1;
          }
        } while (code === SPACE || code === NEWLINE || code === TAB || code === CR || code === FEED);

        currentToken = ['space', css.slice(pos, next)];
        pos = next - 1;
        break;

      case OPEN_SQUARE:
        currentToken = ['[', '[', line, pos - offset];
        break;

      case CLOSE_SQUARE:
        currentToken = [']', ']', line, pos - offset];
        break;

      case OPEN_CURLY:
        currentToken = ['{', '{', line, pos - offset];
        break;

      case CLOSE_CURLY:
        currentToken = ['}', '}', line, pos - offset];
        break;

      case COLON:
        currentToken = [':', ':', line, pos - offset];
        break;

      case SEMICOLON:
        currentToken = [';', ';', line, pos - offset];
        break;

      case OPEN_PARENTHESES:
        prev = buffer.length ? buffer.pop()[1] : '';
        n = css.charCodeAt(pos + 1);
        if (prev === 'url' && n !== SINGLE_QUOTE && n !== DOUBLE_QUOTE && n !== SPACE && n !== NEWLINE && n !== TAB && n !== FEED && n !== CR) {
          next = pos;
          do {
            escaped = false;
            next = css.indexOf(')', next + 1);
            if (next === -1) {
              if (ignore) {
                next = pos;
                break;
              } else {
                unclosed('bracket');
              }
            }
            escapePos = next;
            while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
              escapePos -= 1;
              escaped = !escaped;
            }
          } while (escaped);

          currentToken = ['brackets', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

          pos = next;
        } else {
          next = css.indexOf(')', pos + 1);
          content = css.slice(pos, next + 1);

          if (next === -1 || RE_BAD_BRACKET.test(content)) {
            currentToken = ['(', '(', line, pos - offset];
          } else {
            currentToken = ['brackets', content, line, pos - offset, line, next - offset];
            pos = next;
          }
        }

        break;

      case CLOSE_PARENTHESES:
        currentToken = [')', ')', line, pos - offset];
        break;

      case SINGLE_QUOTE:
      case DOUBLE_QUOTE:
        quote = code === SINGLE_QUOTE ? '\'' : '"';
        next = pos;
        do {
          escaped = false;
          next = css.indexOf(quote, next + 1);
          if (next === -1) {
            if (ignore) {
              next = pos + 1;
              break;
            } else {
              unclosed('string');
            }
          }
          escapePos = next;
          while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
            escapePos -= 1;
            escaped = !escaped;
          }
        } while (escaped);

        content = css.slice(pos, next + 1);
        lines = content.split('\n');
        last = lines.length - 1;

        if (last > 0) {
          nextLine = line + last;
          nextOffset = next - lines[last].length;
        } else {
          nextLine = line;
          nextOffset = offset;
        }

        currentToken = ['string', css.slice(pos, next + 1), line, pos - offset, nextLine, next - nextOffset];

        offset = nextOffset;
        line = nextLine;
        pos = next;
        break;

      case AT:
        RE_AT_END.lastIndex = pos + 1;
        RE_AT_END.test(css);
        if (RE_AT_END.lastIndex === 0) {
          next = css.length - 1;
        } else {
          next = RE_AT_END.lastIndex - 2;
        }

        currentToken = ['at-word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

        pos = next;
        break;

      case BACKSLASH:
        next = pos;
        escape = true;
        while (css.charCodeAt(next + 1) === BACKSLASH) {
          next += 1;
          escape = !escape;
        }
        code = css.charCodeAt(next + 1);
        if (escape && code !== SLASH && code !== SPACE && code !== NEWLINE && code !== TAB && code !== CR && code !== FEED) {
          next += 1;
          if (RE_HEX_ESCAPE.test(css.charAt(next))) {
            while (RE_HEX_ESCAPE.test(css.charAt(next + 1))) {
              next += 1;
            }
            if (css.charCodeAt(next + 1) === SPACE) {
              next += 1;
            }
          }
        }

        currentToken = ['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

        pos = next;
        break;

      default:
        if (code === SLASH && css.charCodeAt(pos + 1) === ASTERISK) {
          next = css.indexOf('*/', pos + 2) + 1;
          if (next === 0) {
            if (ignore) {
              next = css.length;
            } else {
              unclosed('comment');
            }
          }

          content = css.slice(pos, next + 1);
          lines = content.split('\n');
          last = lines.length - 1;

          if (last > 0) {
            nextLine = line + last;
            nextOffset = next - lines[last].length;
          } else {
            nextLine = line;
            nextOffset = offset;
          }

          currentToken = ['comment', content, line, pos - offset, nextLine, next - nextOffset];

          offset = nextOffset;
          line = nextLine;
          pos = next;
        } else {
          RE_WORD_END.lastIndex = pos + 1;
          RE_WORD_END.test(css);
          if (RE_WORD_END.lastIndex === 0) {
            next = css.length - 1;
          } else {
            next = RE_WORD_END.lastIndex - 2;
          }

          currentToken = ['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

          buffer.push(currentToken);

          pos = next;
        }

        break;
    }

    pos++;
    return currentToken;
  }

  function back(token) {
    returned.push(token);
  }

  return {
    back: back,
    nextToken: nextToken,
    endOfFile: endOfFile
  };
}
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
