"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var datetime_1 = require("./datetime");
var log = require("./log");
var util_1 = require("./util");
var TimeUnit;
(function (TimeUnit) {
    TimeUnit.YEAR = 'year';
    TimeUnit.MONTH = 'month';
    TimeUnit.DAY = 'day';
    TimeUnit.DATE = 'date';
    TimeUnit.HOURS = 'hours';
    TimeUnit.MINUTES = 'minutes';
    TimeUnit.SECONDS = 'seconds';
    TimeUnit.MILLISECONDS = 'milliseconds';
    TimeUnit.YEARMONTH = 'yearmonth';
    TimeUnit.YEARMONTHDATE = 'yearmonthdate';
    TimeUnit.YEARMONTHDATEHOURS = 'yearmonthdatehours';
    TimeUnit.YEARMONTHDATEHOURSMINUTES = 'yearmonthdatehoursminutes';
    TimeUnit.YEARMONTHDATEHOURSMINUTESSECONDS = 'yearmonthdatehoursminutesseconds';
    // MONTHDATE always include 29 February since we use year 0th (which is a leap year);
    TimeUnit.MONTHDATE = 'monthdate';
    TimeUnit.HOURSMINUTES = 'hoursminutes';
    TimeUnit.HOURSMINUTESSECONDS = 'hoursminutesseconds';
    TimeUnit.MINUTESSECONDS = 'minutesseconds';
    TimeUnit.SECONDSMILLISECONDS = 'secondsmilliseconds';
    TimeUnit.QUARTER = 'quarter';
    TimeUnit.YEARQUARTER = 'yearquarter';
    TimeUnit.QUARTERMONTH = 'quartermonth';
    TimeUnit.YEARQUARTERMONTH = 'yearquartermonth';
    TimeUnit.UTCYEAR = 'utcyear';
    TimeUnit.UTCMONTH = 'utcmonth';
    TimeUnit.UTCDAY = 'utcday';
    TimeUnit.UTCDATE = 'utcdate';
    TimeUnit.UTCHOURS = 'utchours';
    TimeUnit.UTCMINUTES = 'utcminutes';
    TimeUnit.UTCSECONDS = 'utcseconds';
    TimeUnit.UTCMILLISECONDS = 'utcmilliseconds';
    TimeUnit.UTCYEARMONTH = 'utcyearmonth';
    TimeUnit.UTCYEARMONTHDATE = 'utcyearmonthdate';
    TimeUnit.UTCYEARMONTHDATEHOURS = 'utcyearmonthdatehours';
    TimeUnit.UTCYEARMONTHDATEHOURSMINUTES = 'utcyearmonthdatehoursminutes';
    TimeUnit.UTCYEARMONTHDATEHOURSMINUTESSECONDS = 'utcyearmonthdatehoursminutesseconds';
    // MONTHDATE always include 29 February since we use year 0th (which is a leap year);
    TimeUnit.UTCMONTHDATE = 'utcmonthdate';
    TimeUnit.UTCHOURSMINUTES = 'utchoursminutes';
    TimeUnit.UTCHOURSMINUTESSECONDS = 'utchoursminutesseconds';
    TimeUnit.UTCMINUTESSECONDS = 'utcminutesseconds';
    TimeUnit.UTCSECONDSMILLISECONDS = 'utcsecondsmilliseconds';
    TimeUnit.UTCQUARTER = 'utcquarter';
    TimeUnit.UTCYEARQUARTER = 'utcyearquarter';
    TimeUnit.UTCQUARTERMONTH = 'utcquartermonth';
    TimeUnit.UTCYEARQUARTERMONTH = 'utcyearquartermonth';
})(TimeUnit = exports.TimeUnit || (exports.TimeUnit = {}));
/** Time Unit that only corresponds to only one part of Date objects. */
var LOCAL_SINGLE_TIMEUNIT_INDEX = {
    year: 1,
    quarter: 1,
    month: 1,
    day: 1,
    date: 1,
    hours: 1,
    minutes: 1,
    seconds: 1,
    milliseconds: 1
};
exports.TIMEUNIT_PARTS = util_1.flagKeys(LOCAL_SINGLE_TIMEUNIT_INDEX);
function isLocalSingleTimeUnit(timeUnit) {
    return !!LOCAL_SINGLE_TIMEUNIT_INDEX[timeUnit];
}
exports.isLocalSingleTimeUnit = isLocalSingleTimeUnit;
var UTC_SINGLE_TIMEUNIT_INDEX = {
    utcyear: 1,
    utcquarter: 1,
    utcmonth: 1,
    utcday: 1,
    utcdate: 1,
    utchours: 1,
    utcminutes: 1,
    utcseconds: 1,
    utcmilliseconds: 1
};
function isUtcSingleTimeUnit(timeUnit) {
    return !!UTC_SINGLE_TIMEUNIT_INDEX[timeUnit];
}
exports.isUtcSingleTimeUnit = isUtcSingleTimeUnit;
var LOCAL_MULTI_TIMEUNIT_INDEX = {
    yearquarter: 1,
    yearquartermonth: 1,
    yearmonth: 1,
    yearmonthdate: 1,
    yearmonthdatehours: 1,
    yearmonthdatehoursminutes: 1,
    yearmonthdatehoursminutesseconds: 1,
    quartermonth: 1,
    monthdate: 1,
    hoursminutes: 1,
    hoursminutesseconds: 1,
    minutesseconds: 1,
    secondsmilliseconds: 1
};
var UTC_MULTI_TIMEUNIT_INDEX = {
    utcyearquarter: 1,
    utcyearquartermonth: 1,
    utcyearmonth: 1,
    utcyearmonthdate: 1,
    utcyearmonthdatehours: 1,
    utcyearmonthdatehoursminutes: 1,
    utcyearmonthdatehoursminutesseconds: 1,
    utcquartermonth: 1,
    utcmonthdate: 1,
    utchoursminutes: 1,
    utchoursminutesseconds: 1,
    utcminutesseconds: 1,
    utcsecondsmilliseconds: 1
};
var UTC_TIMEUNIT_INDEX = tslib_1.__assign({}, UTC_SINGLE_TIMEUNIT_INDEX, UTC_MULTI_TIMEUNIT_INDEX);
function isUTCTimeUnit(t) {
    return !!UTC_TIMEUNIT_INDEX[t];
}
exports.isUTCTimeUnit = isUTCTimeUnit;
function getLocalTimeUnit(t) {
    return t.substr(3);
}
exports.getLocalTimeUnit = getLocalTimeUnit;
var TIMEUNIT_INDEX = tslib_1.__assign({}, LOCAL_SINGLE_TIMEUNIT_INDEX, UTC_SINGLE_TIMEUNIT_INDEX, LOCAL_MULTI_TIMEUNIT_INDEX, UTC_MULTI_TIMEUNIT_INDEX);
exports.TIMEUNITS = util_1.flagKeys(TIMEUNIT_INDEX);
function isTimeUnit(t) {
    return !!TIMEUNIT_INDEX[t];
}
exports.isTimeUnit = isTimeUnit;
var SET_DATE_METHOD = {
    year: 'setFullYear',
    month: 'setMonth',
    date: 'setDate',
    hours: 'setHours',
    minutes: 'setMinutes',
    seconds: 'setSeconds',
    milliseconds: 'setMilliseconds',
    // Day and quarter have their own special cases
    quarter: null,
    day: null,
};
/**
 * Converts a date to only have the measurements relevant to the specified unit
 * i.e. ('yearmonth', '2000-12-04 07:58:14') -> '2000-12-01 00:00:00'
 * Note: the base date is Jan 01 1900 00:00:00
 */
function convert(unit, date) {
    var isUTC = isUTCTimeUnit(unit);
    var result = isUTC ?
        // start with uniform date
        new Date(Date.UTC(0, 0, 1, 0, 0, 0, 0)) :
        new Date(0, 0, 1, 0, 0, 0, 0);
    for (var _i = 0, TIMEUNIT_PARTS_1 = exports.TIMEUNIT_PARTS; _i < TIMEUNIT_PARTS_1.length; _i++) {
        var timeUnitPart = TIMEUNIT_PARTS_1[_i];
        if (containsTimeUnit(unit, timeUnitPart)) {
            switch (timeUnitPart) {
                case TimeUnit.DAY:
                    throw new Error('Cannot convert to TimeUnits containing \'day\'');
                case TimeUnit.QUARTER: {
                    var _a = dateMethods('month', isUTC), getDateMethod_1 = _a.getDateMethod, setDateMethod_1 = _a.setDateMethod;
                    // indicate quarter by setting month to be the first of the quarter i.e. may (4) -> april (3)
                    result[setDateMethod_1]((Math.floor(date[getDateMethod_1]() / 3)) * 3);
                    break;
                }
                default:
                    var _b = dateMethods(timeUnitPart, isUTC), getDateMethod = _b.getDateMethod, setDateMethod = _b.setDateMethod;
                    result[setDateMethod](date[getDateMethod]());
            }
        }
    }
    return result;
}
exports.convert = convert;
function dateMethods(singleUnit, isUtc) {
    var rawSetDateMethod = SET_DATE_METHOD[singleUnit];
    var setDateMethod = isUtc ? 'setUTC' + rawSetDateMethod.substr(3) : rawSetDateMethod;
    var getDateMethod = 'get' + (isUtc ? 'UTC' : '') + rawSetDateMethod.substr(3);
    return { setDateMethod: setDateMethod, getDateMethod: getDateMethod };
}
function getTimeUnitParts(timeUnit) {
    return exports.TIMEUNIT_PARTS.reduce(function (parts, part) {
        if (containsTimeUnit(timeUnit, part)) {
            return parts.concat(part);
        }
        return parts;
    }, []);
}
exports.getTimeUnitParts = getTimeUnitParts;
/** Returns true if fullTimeUnit contains the timeUnit, false otherwise. */
function containsTimeUnit(fullTimeUnit, timeUnit) {
    var index = fullTimeUnit.indexOf(timeUnit);
    return index > -1 &&
        (timeUnit !== TimeUnit.SECONDS ||
            index === 0 ||
            fullTimeUnit.charAt(index - 1) !== 'i' // exclude milliseconds
        );
}
exports.containsTimeUnit = containsTimeUnit;
/**
 * Returns Vega expresssion for a given timeUnit and fieldRef
 */
function fieldExpr(fullTimeUnit, field) {
    var fieldRef = "datum" + util_1.accessPath(field);
    var utc = isUTCTimeUnit(fullTimeUnit) ? 'utc' : '';
    function func(timeUnit) {
        if (timeUnit === TimeUnit.QUARTER) {
            // quarter starting at 0 (0,3,6,9).
            return "(" + utc + "quarter(" + fieldRef + ")-1)";
        }
        else {
            return "" + utc + timeUnit + "(" + fieldRef + ")";
        }
    }
    var d = exports.TIMEUNIT_PARTS.reduce(function (dateExpr, tu) {
        if (containsTimeUnit(fullTimeUnit, tu)) {
            dateExpr[tu] = func(tu);
        }
        return dateExpr;
    }, {});
    return datetime_1.dateTimeExpr(d);
}
exports.fieldExpr = fieldExpr;
/**
 * returns the signal expression used for axis labels for a time unit
 */
function formatExpression(timeUnit, field, shortTimeLabels, isUTCScale) {
    if (!timeUnit) {
        return undefined;
    }
    var dateComponents = [];
    var expression = '';
    var hasYear = containsTimeUnit(timeUnit, TimeUnit.YEAR);
    if (containsTimeUnit(timeUnit, TimeUnit.QUARTER)) {
        // special expression for quarter as prefix
        expression = "'Q' + quarter(" + field + ")";
    }
    if (containsTimeUnit(timeUnit, TimeUnit.MONTH)) {
        // By default use short month name
        dateComponents.push(shortTimeLabels !== false ? '%b' : '%B');
    }
    if (containsTimeUnit(timeUnit, TimeUnit.DAY)) {
        dateComponents.push(shortTimeLabels ? '%a' : '%A');
    }
    else if (containsTimeUnit(timeUnit, TimeUnit.DATE)) {
        dateComponents.push('%d' + (hasYear ? ',' : '')); // add comma if there is year
    }
    if (hasYear) {
        dateComponents.push(shortTimeLabels ? '%y' : '%Y');
    }
    var timeComponents = [];
    if (containsTimeUnit(timeUnit, TimeUnit.HOURS)) {
        timeComponents.push('%H');
    }
    if (containsTimeUnit(timeUnit, TimeUnit.MINUTES)) {
        timeComponents.push('%M');
    }
    if (containsTimeUnit(timeUnit, TimeUnit.SECONDS)) {
        timeComponents.push('%S');
    }
    if (containsTimeUnit(timeUnit, TimeUnit.MILLISECONDS)) {
        timeComponents.push('%L');
    }
    var dateTimeComponents = [];
    if (dateComponents.length > 0) {
        dateTimeComponents.push(dateComponents.join(' '));
    }
    if (timeComponents.length > 0) {
        dateTimeComponents.push(timeComponents.join(':'));
    }
    if (dateTimeComponents.length > 0) {
        if (expression) {
            // Add space between quarter and main time format
            expression += " + ' ' + ";
        }
        // We only use utcFormat for utc scale
        // For utc time units, the data is already converted as a part of timeUnit transform.
        // Thus, utc time units should use timeFormat to avoid shifting the time twice.
        if (isUTCScale) {
            expression += "utcFormat(" + field + ", '" + dateTimeComponents.join(' ') + "')";
        }
        else {
            expression += "timeFormat(" + field + ", '" + dateTimeComponents.join(' ') + "')";
        }
    }
    // If expression is still an empty string, return undefined instead.
    return expression || undefined;
}
exports.formatExpression = formatExpression;
function normalizeTimeUnit(timeUnit) {
    if (timeUnit !== 'day' && timeUnit.indexOf('day') >= 0) {
        log.warn(log.message.dayReplacedWithDate(timeUnit));
        return timeUnit.replace('day', 'date');
    }
    return timeUnit;
}
exports.normalizeTimeUnit = normalizeTimeUnit;
//# sourceMappingURL=data:application/json;base64,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