/*	$OpenBSD: fmemopentest.c,v 1.4 2020/08/17 16:17:39 millert Exp $ */

/*
 * Copyright (c) 2011 Martin Pieuchot <mpi@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

int
simpletest(void)
{
	FILE	*s1, *s2;
	char	 string[] = "fmemopen test string!";
	char	 buffer[1024], *buf = NULL;
	size_t	 len;
	int	 c, failures = 0;

	s1 = fmemopen(string, strlen(string) + 1, "r");
	if (s1 == NULL) {
		warn("unable to open a stream s1");
		return (1);
	}

	s2 = fmemopen(buf, 22, "w+");
	if (s2 == NULL) {
		warn("unable to create a stream s2");
		fclose(s1);
		return (1);
	}

	while ((c = fgetc(s1)) != EOF)
		fputc(c, s2);

	if (ftell(s2) != strlen(string) + 1) {
		warnx("failed copy test (1)");
		failures++;
	}
	fclose(s1);

	fseek(s2, 0, SEEK_SET);
	if (ftell(s2) != 0) {
		warnx("failed seek test (2)");
		failures++;
	}

	len = fread(buffer, 1, sizeof(buffer) - 1, s2);
	if (len != strlen(string) + 1) {
		warnx("failed read test (3) %zu != %zu",
		    len, strlen(string) + 1);
		failures++;
	}

	return (failures);
}

int
appendtest(const char *mode)
{
	FILE	*s1;
	char	 string[] = "hello\0 test number 2";
	char	 buffer[256];
	size_t	 len;
	int	 failures = 0;

	s1 = fmemopen(string, 19, mode);
	if (s1 == NULL)
		return (1);

	fseek(s1, 0, SEEK_SET);
	if (ftell(s1) != 0) {
		warnx("failed seek test [%s] (4)", mode);
		failures++;
	}

	/* write should append even though seek position is 0 */
	len = fwrite(" world", 1, 6, s1);
	if (len != 6) {
		warnx("failed write test [%s] (5)", mode);
		failures++;
	}

	if (ftell(s1) != strlen("hello world")) {
		warnx("failed seek test [%s] (6)", mode);
		failures++;
	}

	if (mode[1] == '+') {
		fseek(s1, 0, SEEK_SET);
		len = fread(buffer, 1, sizeof(buffer) - 1, s1);
		if (len == 0 || strncmp(string, buffer, len)) {
			warnx("failed compare test [%s] (7)", mode);
			failures++;
		}
	}

	if (strcmp(string, "hello world")) {
		warnx("failed compare test [%s] (8)", mode);
		failures++;
	}

	if (strcmp(string + strlen(string) + 1, "number 2")) {
		warnx("failed compare test [%s] (9)", mode);
		failures++;
	}

	return (failures);
}

int
updatetest(void)
{
	FILE	*s1;
	char	 string[] = "bah, what a day";
	char	 buffer[256];
	size_t	 len;
	int	 failures = 0;

	s1 = fmemopen(string, 19, "r+");
	if (s1 == NULL)
		return (1);

	if (ftell(s1) != 0) {
		warnx("failed seek test (10)");
		failures++;
	}

	len = fwrite("oh frabjous", 1, 11, s1);
	if (len != 11) {
		warnx("failed write test (11)");
		failures++;
	}

	fseek(s1, 0, SEEK_SET);
	len = fread(buffer, 1, sizeof(buffer) - 1, s1);
	if (len == 0 || strncmp(string, buffer, len)) {
		warnx("failed compare test (12)");
		failures++;
	}

	if (strcmp(string, "oh frabjous day")) {
		warnx("failed compare test (13)");
		failures++;
	}

	return (failures);
}

int
writetest(const char *mode)
{
	FILE	*s1;
	char	 string[] = "super test number 3";
	char	 buffer[256];
	size_t	 len, slen;
	int	 failures = 0;

	slen = strlen(string) + 1;

	s1 = fmemopen(string, slen, mode);
	if (s1 == NULL)
		return (1);

	len = fwrite("short", 1, 5, s1);
	if (len != strlen("short")) {
		warnx("failed write test [%s] (14)", mode);
		failures++;
	}
	fclose(s1);

	s1 = fmemopen(string, slen, "r");
	if (s1 == NULL) {
		warnx("failed open test [%s] (15)", mode);
		failures++;
	}

	len = fread(buffer, 1, sizeof(buffer) - 1, s1);
	if (len == 0 || strncmp(string, buffer, len)) {
		warnx("failed compare test [%s] (16)", mode);
		failures++;
	}

	if (strcmp(string, "short")) {
		warnx("failed compare test [%s] (17)", mode);
		failures++;
	}

	if (strcmp(string + strlen(string) + 1, "test number 3")) {
		warnx("failed compare test [%s] (18)", mode);
		failures++;
	}

	return (failures);
}

int
seektest(void)
{
	FILE	*s1;
	char	 string[] = "long string for testing seek";
	size_t	 len, slen;
	int	 failures = 0;

	slen = strlen(string) + 1;

	s1 = fmemopen(string, slen, "r");
	if (s1 == NULL)
		return (1);

	if (fseek(s1, 8, SEEK_SET) != 0) {
		warnx("failed to fseek. (19)");
		failures++;
	}

	if (ftell(s1) != 8) {
		warnx("failed seek test. (20)");
		failures++;
	}

	/* Try to seek backward */
	if (fseek(s1, -1, SEEK_CUR) != 0) {
		warnx("failed to fseek. (21)");
		failures++;
	}

	if (ftell(s1) != 7) {
		warnx("failed seeking backward. (22)");
		failures++;
	}

	return (failures);
}

int
main(void)
{
	int	 failures = 0;

	failures += simpletest();
	failures += appendtest("a");
	failures += appendtest("a+");
	failures += updatetest();
	failures += writetest("w");
	failures += writetest("w+");
	failures += seektest();

 	return (failures);
}
