package gnu.crypto.keyring;  // -*- c-basic-offset: 3 -*-

// ---------------------------------------------------------------------------
// $Id: Entry.java,v 1.4.2.1 2004/02/09 21:42:38 rsdio Exp $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    59 Temple Place - Suite 330,
//    Boston, MA 02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
//
// ---------------------------------------------------------------------------

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * An immutable class representing a single entry in a keyring.
 *
 * @version $Revision: 1.4.2.1 $
 */
public abstract class Entry {

   // Fields.
   // ------------------------------------------------------------------------

   /** This entry's type identifier. */
   protected int type;

   /** This entry's property set. */
   protected Properties properties;

   /** This entry's payload. */
   protected byte[] payload;

   // Constructor.
   // ------------------------------------------------------------------------

   /**
    * Creates a new Entry.
    *
    * @param type This entry's type.
    * @param properties This entry's properties.
    * @throws IllegalArgumentException If the properties argument is null,
    *   or if the type is out of range.
    */
   protected Entry(int type, Properties properties) {
      if (type < 0 || type > 255) {
         throw new IllegalArgumentException("invalid packet type");
      }
      if (properties == null) {
         throw new IllegalArgumentException("no properties");
      }
      this.type = type;
      this.properties = (Properties) properties.clone();
   }

   /**
    * Default constructor for use by subclasses.
    */
   protected Entry() {
   }

   // Instance methods.
   // ------------------------------------------------------------------------

   /**
    * Returns this entry's properties object. The properties are cloned before
    * being returned.
    *
    * @return The properties.
    */
   public Properties getProperties() {
      return (Properties) properties.clone();
   }

   /**
    * Returns this entry's payload data, or null if
    */
   public byte[] getPayload() {
      if (payload == null)
         return null;
      return (byte[]) payload.clone();
   }

   /**
    * This method is called when this entry needs to be written to an
    * output stream.
    *
    * @param out The stream to write to.
    * @throws IOException If an I/O exception occurs.
    */
   public void encode(DataOutputStream out) throws IOException
   {
      if (payload == null) {
         encodePayload();
      }
      if (out == null) {
         return;
      }
      out.write(type);
      properties.encode(out);
      out.writeInt(payload.length);
      out.write(payload);
   }

   /**
    * Generic decoding method, which simply decodes the properties field
    * and reads the payload field.
    *
    * @param in The input data stream.
    * @throws IOException If an I/O error occurs.
    */
   protected void defaultDecode(DataInputStream in) throws IOException {
      properties = new Properties();
      properties.decode(in);
      int len = in.readInt();
      if (len < 0) {
         throw new IOException("corrupt length");
      }
      payload = new byte[len];
      in.readFully(payload);
   }

   // Abstract methods.
   // ------------------------------------------------------------------------

   /**
    * This method is called of subclasses when the payload data needs to be
    * created.
    *
    * @throws IOException If an encoding error occurs.
    */
   protected abstract void encodePayload() throws IOException;
}
