/*
 * Routines for updating screen display
 *
  gsumi version 0.5

  Copyright 1997 Owen Taylor <owt1@cornell.edu>

  Heavily based on

  xink version 0.02

  Copyright 1997 Raph Levien <raph@acm.org>

  This code is free for commercial and non-commercial use or
  redistribution, as long as the source code release, startup screen,
  or product packaging includes this copyright notice.

*/

#include "gsumi.h"
#include "region.h"

/* The transform from display coordinates into bitmap coordinates.
   display_xform2 is the transform associated with the redraw of
   render_region2 - changes to it are deferred until that render
   completes. On quiescence, display_xform and display_xform2 will
   be equal. */
xform display_xform, display_xform2;

/* The image for handling expose events */
GdkImage *img = NULL;
int image_width;
int image_height;

/* These may be 32, 16, or 8 bpp pixels, depending on bpp. */
gulong gray_pixels[256];
gulong red_pixel;
gulong maroon_pixel;

/* The region (in screen coordinates) that needs to be rendered from the
   bitmap to the screen image.

   render_region2 is the region currently being rendered. While it's
   being rendered, new modifications are queued up in render_region.
   The union of these two regions represents the work to be done.
*/
region *render_region, *render_region2;

/* The region of the screen image (i.e. ximage) that needs to actually
   be drawn on the screen (using XPutImage or its shared memory variant.
   As above, draw_region2 is the region currently being drawn. */
region *draw_region, *draw_region2;

/* This rectangle represents the size of the pane (i.e. the window
   minus the scrollbars and toolbar. */
rect pane_rec;

#define CURSOR_NONE 0
#define CURSOR_CROSS 1
#define CURSOR_CIRC 2
point cursor_pos, cursor_pos2;
int cursor_icon, cursor_icon2;

void create_ximage (GtkWidget *area, int width, int height) {

  if (img != NULL)
    {
      if ((image_width == width) && (image_height == height))
	return;
      else
	gdk_image_destroy(img);
    }

  image_width = width;
  image_height = height;

  pane_rec.x0 = 0;
  pane_rec.y0 = 0;
  pane_rec.x1 = image_width;
  pane_rec.y1 = image_height;

  img = gdk_image_new(GDK_IMAGE_FASTEST,
		      gdk_window_get_visual(area->window),
		      image_width,
		      image_height);

  switch (img->bpp) {
  case 1:
    memset (img->mem, gray_pixels[255], img->bpl * image_height);
    break;
  case 2:
    memset (img->mem, 255, img->bpl * image_height);
    break;
  case 3:
    memset (img->mem, 255, img->bpl * image_height);
    break;
  case 4:
    {
      int i;

      /* what if we have more than 4 byte alignment of rows??? */
      for (i = 0; i < image_width * image_height; i++)
	  ((int32 *)img->mem)[i] = 0xffffff;
      break;
    }
  default:
    fprintf(stderr,"Can't handle %d byte/pixel pixmaps\n",img->bpp);
    exit(1);
  }
}

/* There is duplication in the next two pairs of routines. Perhaps
   pairs of regions (as in render_region, render_region2) should
   become an abstract data type. For the time being, though, the
   amount of duplicated code is so small I'm not worrying about it.
*/

/* Flow draw_region to draw_region2 if the latter is empty. */
void flow_draw_region (void) {
  region *tmp;

  if (region_empty (draw_region2)) {
    tmp = draw_region2;
    draw_region2 = draw_region;
    draw_region = tmp;
    if (!region_empty(draw_region2))
      start_idle_update();
  }
}

/* Add a rectangle to the draw region. */
void expose_rect (rect *rec) {
  draw_region = region_union (draw_region,
			      region_minus (region_from_rect (rec),
					    region_pin (draw_region2)));
  flow_draw_region ();
}

#define CURSOR_LEN 8

void rect_from_cursor (rect *dst, const point *pos, int icon) {
  if (!icon) {
    dst->x0 = dst->x1 = dst->y0 = dst->y1 = 0;
  } else {
    rect rec;
    rec.x0 = pos->x - CURSOR_LEN;
    rec.y0 = pos->y - CURSOR_LEN;
    rec.x1 = pos->x + CURSOR_LEN - 1;
    rec.y1 = pos->y + CURSOR_LEN - 1;
    rect_intersect (dst, &rec, &pane_rec);
  }
}

/* Flow render_region to render_region2 if the latter is empty. Also,
   if the associated display_xform has changed, then designate the
   entire pane to be re-rendered. 

   Also, some of the cursor drawing logic is here.

*/
void flow_render_region (void) {
  region *tmp;

  if (region_empty (render_region2)) {
    tmp = render_region2;
    render_region2 = render_region;
    render_region = tmp;
    if (display_xform.scale != display_xform2.scale ||
	display_xform.off_x != display_xform2.off_x ||
	display_xform.off_y != display_xform2.off_y ) {
      display_xform2 = display_xform;
      region_drop (render_region2);
      render_region2 = region_from_rect (&pane_rec);
    }
    if (cursor_pos.x != cursor_pos2.x ||
	cursor_pos.y != cursor_pos2.y ||
	cursor_icon != cursor_icon2) {
      rect cursor_rec;
      rect cursor_rec2;

      rect_from_cursor (&cursor_rec, &cursor_pos, cursor_icon);
      rect_from_cursor (&cursor_rec2, &cursor_pos2, cursor_icon2);
      render_region2 = region_union (render_region2,
				     region_from_rect (&cursor_rec));
      render_region2 = region_union (render_region2,
				     region_from_rect (&cursor_rec2));
      cursor_pos2 = cursor_pos;
      cursor_icon2 = cursor_icon;
    }
  }
  if (!region_empty(render_region2))
    start_idle_update();
}

/* Add a rectangle to the render region. This routine clips the render
   region to the window. The argument is in window coordinates. */
void add_to_render_region (rect *rec) {
  rect clip_rec;

  rect_intersect (&clip_rec, rec, &pane_rec);
  render_region = region_union (render_region,
				region_minus (region_from_rect (&clip_rec),
					      region_pin (render_region2)));
  flow_render_region ();
}

static void gray_run (int y, int x_min, int x_max)
{
  int i;
  uint8 *img_line = (uint8 *)img->mem + y*img->bpl;
  switch (img->bpp) {
  case 1:
    for (i = x_min; i < x_max; i++)
      ((uint8 *)img_line)[i] = gray_pixels[127];
    break;
  case 2:
    for (i = x_min; i < x_max; i++)
      ((uint16 *)img_line)[i] = gray_pixels[127];
    break;
  case 3:
    for (i = x_min; i < x_max; i++) 
      {
	((uint8 *)img_line)[i * 3] = gray_pixels[127] >> 16;
	((uint8 *)img_line)[i * 3 + 1] = (gray_pixels[127] >> 8) & 0xff;
	((uint8 *)img_line)[i * 3 + 2] = gray_pixels[127] & 0xff;
      }
  case 4:
    for (i = x_min; i < x_max; i++)
      ((uint32 *)img_line)[i] = gray_pixels[127];
    break;
  }
}

#define MAX_IMAGE_WIDTH 2048

/* we set the portion of the background that doesn't contain
   the image to 50% gray. There is some error in the way
   we antialias the edges - we act as if the background intensity
   was scale*(scale/2) */

/* Render a rectangle (image coordinates) to the ximage. */
void bitmap_render_to_ximage (int x, int y, int width, int height) {
  int i, j, jj, k;
  unsigned char accum[MAX_IMAGE_WIDTH];
  int pos, pos_1, pos_2;
  int32 grays[256];
  int g;

  int maxg;
  int maxpos, minpos;
  int k1, k2;
  int y_min, y_max;
  int bm_x_max;
  int bm_y_max;
  int scale;
  int off_x, off_y;

  uint8 *img_line;

#ifdef DEBUG_RENDER
  printf ("rendering %d %d %d %d\n", x, y, width, height);
#endif

  scale = display_xform2.scale;
  off_x = display_xform2.off_x;
  off_y = display_xform2.off_y;
  maxg = scale * scale;
  for (i = 0; i <= maxg; i++) {
    g = ((maxg - i) * 255) / maxg;
    grays[i] = gray_pixels[g];
  }
  minpos = x * scale;
  maxpos = (x + width) * scale;
  y_min = y;
  y_max = y + height;

  bm_x_max = 1+(bitmap->width-1) / scale;
  if (bm_x_max < x) 
    bm_x_max = x;
  if (bm_x_max > x+width) 
    bm_x_max = x+width;
  bm_y_max = 1+(bitmap->height-1) / scale;
  if (bm_y_max < y)
    bm_y_max = y;
  if (bm_y_max > y+height) 
    bm_y_max = y+height;

  for (j = y_min; j < bm_y_max; j++) {
    for (i = x; i < bm_x_max; i++)
      accum[i] = 0;
    if (scale == 4) {
      /* version specialized to scale == 4 */
      for (jj = j * 4 + off_y; jj < (j + 1) * 4 + off_y; jj++)
	if (jj >= 0 && jj < bitmap->height) {
	  /* accumulate a runlength line into the accum buffer */
	  pos = -off_x;
	  for (i = 0; bitmap->lines[jj][i]; i++) {
	    pos_1 = pos + WHITE_X(bitmap->lines[jj][i]);
	    pos_2 = pos_1 + BLACK_X(bitmap->lines[jj][i]);
	    if (pos_1 < maxpos && pos_2 >= minpos) {
	      k1 = pos_1 >> 2;
	      if (k1 < x) k1 = x;
	      else accum[k1] -= (pos_1 & 3);
	      k2 = pos_2 >> 2;
	      if (k2 >= x + width) k2 = x + width;
	      else accum[k2] += (pos_2 & 3);
	      for (k = k1; k < k2; k++)
		accum[k] += 4;
	    }
	    pos = pos_2;
	  }
	  /* fix up remainder of last pixel */
	  if (bitmap->width % 4)
	    accum[bitmap->width / 4] += (4 - bitmap->width % 4)/2;
	} else if (jj >= bitmap->height) { /* set rest of lines to mid gray */
	  for (i=x;i<bm_x_max;i++)
	    accum[i] += 2;
	}
    } else {
      for (jj = j * scale + off_y; jj < (j + 1) * scale + off_y; jj++)
	if (jj >= 0 && jj < bitmap->height) {
	  /* accumulate a runlength line into the accum buffer */
	  pos = -off_x;
	  for (i = 0; bitmap->lines[jj][i]; i++) {
	    pos_1 = pos + WHITE_X(bitmap->lines[jj][i]);
	    pos_2 = pos_1 + BLACK_X(bitmap->lines[jj][i]);
	    if (pos_1 < maxpos && pos_2 >= minpos) {
	      k1 = pos_1 / scale;
	      if (k1 < x) k1 = x;
	      else accum[k1] -= (pos_1 % scale);
	      k2 = pos_2 / scale;
	      if (k2 >= x + width) k2 = x + width;
	      else accum[k2] += (pos_2 % scale);
	      for (k = k1; k < k2; k++)
		accum[k] += scale;
	    }
	    pos = pos_2;
	  }
	  /* fix up remainder of last pixel */
	  if (bitmap->width % scale)
	    accum[bitmap->width / scale] += (scale - bitmap->width % scale)/2;
	} else if (jj >= bitmap->height) { /* set rest of lines to mid gray */
	  for (i=x;i<bm_x_max;i++)
	    accum[i] += scale/2;
	}
    }
    img_line = (uint8 *)img->mem + j*img->bpl;
    switch (img->bpp) {
    case 1:
      for (i = x; i < bm_x_max; i++)
	((uint8 *)img_line)[i] = grays[(int)accum[i]];
      break;
    case 2:
      for (i = x; i < bm_x_max; i++)
	((uint16 *)img_line)[i] = grays[(int)accum[i]];
      break;
    case 3:
      for (i = x; i < bm_x_max; i++) 
	{
	  ((uint8 *)img_line)[i * 3] = grays[(int)accum[i]] >> 16;
	  ((uint8 *)img_line)[i * 3 + 1] = (grays[(int)accum[i]] >> 8) & 0xff;
	  ((uint8 *)img_line)[i * 3 + 2] = grays[(int)accum[i]] & 0xff;
	}
      break;
    case 4:
      for (i = x; i < bm_x_max; i++)
	((uint32 *)img_line)[i] = grays[(int)accum[i]];
      break;
    }
    gray_run(j,bm_x_max,x+width);
  }
  for (j=bm_y_max;j<y+height;j++)
    gray_run(j,x,x+width);
}

int32 cursor_bits[2][15] = {
#if 0
  {0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
   0x7fff, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80},
#else
  {0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x0,
   0x7e3f, 0x0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80},
#endif
#if 0
  {0x3e0, 0xc18, 0x1004, 0x2002, 0x2002, 0x4001, 0x4001,
   0x4001, 0x4001, 0x4001, 0x2002, 0x2002, 0x1004, 0xc18, 0x3e0}
#else
  {0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
   0x7f7f, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80}
#endif
};
/* Draw the cursor on the ximage, clipped to the given rectangle. */
void cursor_render (const rect *clip_rect) {
  int x, y;
  int k;
  int bmap, mask;
  uint8 *img_line;
  int off_x, off_y;

  if (cursor_icon2 == CURSOR_NONE) return;
  off_x = clip_rect->x0 - cursor_pos2.x + CURSOR_LEN;
  off_y = clip_rect->y0 - cursor_pos2.y + CURSOR_LEN;
  for (y = clip_rect->y0; y < clip_rect->y1; y++) {
    img_line = (uint8 *)img->mem + y * img->bpl;
    k = clip_rect->x0;
    bmap = cursor_bits[cursor_icon2 - 1][off_y];
    mask = 1 << off_x;
    for (x = clip_rect->x0; x < clip_rect->x1; x++) {
      if (bmap & mask)
	switch (img->bpp) {
	case 1:
	  ((uint8 *)img_line)[k] = red_pixel;
	  break;
	case 2:
	  ((uint16 *)img_line)[k] = red_pixel;
	  break;
        case 3:
          ((uint8 *)img_line)[k * 3] = red_pixel >> 16;
          ((uint8 *)img_line)[k * 3 + 1] = (red_pixel >> 8) & 0xff;
          ((uint8 *)img_line)[k * 3 + 2] = red_pixel & 0xff;
          break;
	case 4:
	  ((uint32 *)img_line)[k] = red_pixel;
	  break;
	}
      k++;
      mask <<= 1;
    }
    off_y++;
  }
}

/* Render a chunk of the render region. This routine contains the main
   dispatch for rendering all of the different parts of the window. */
int render_chunk (GtkWidget *area) {
  rect rec, clip_rec, cursor_rec;

  if (region_empty (render_region2)) 
    return 0;
  region_tile (&rec, render_region2, image_width, 32, 1000);

  /* Render the bitmap. */
  rect_intersect (&clip_rec, &rec, &pane_rec);
  if (!rect_empty (&clip_rec))
    bitmap_render_to_ximage (clip_rec.x0, clip_rec.y0,
			     clip_rec.x1 - clip_rec.x0,
			     clip_rec.y1 - clip_rec.y0);

  /* Render the cursor. */
  rect_from_cursor (&cursor_rec, &cursor_pos2, cursor_icon2);
  rect_intersect (&clip_rec, &cursor_rec, &rec);
  if (!rect_empty (&clip_rec))
    cursor_render (&clip_rec);

  expose_rect (&rec);
  render_region2 = region_minus (render_region2, region_from_rect (&rec));
  flow_render_region ();

  return !region_empty (render_region2);
}

/* Call this routine with an updated rectangle before modifying the
   bitmap itself. It adds the modified rectangle into the render
   region. It does not call undo_update. Routines which draw into the
   bitmap should call undo_update as well. The argument is in bitmap
   coordinates. */
void bitmap_update (rect *rec) {
  rect new_rec, clip_rec;
  xform inv_x;

  xform_inv (&inv_x, &display_xform2);
  xform_rect (&new_rec, &inv_x, rec);
  rect_intersect (&clip_rec, &new_rec, &pane_rec);
  add_to_render_region (&clip_rec);
}

/* x and y are in screen coordinates. */
void update_cursor (double x, double y, int prox) {
  cursor_pos.x = floor (x + 0.5);
  cursor_pos.y = floor (y + 0.5);
  cursor_icon = CURSOR_CROSS * prox; /* A hack! */
  flow_render_region ();
}

/* This routine updates the display xform. First, it sanitychecks the
   new xform, putting it in bounds. Then, it updates display_xform.
   Finally, it performs a flow_render_region, which has the effect
   of putting the repaint in the queue if the render region is empty. */
void update_display_xform (xform *new) {

  if (new->scale > 15) new->scale = 15;
  if (new->scale < 1) new->scale = 1;
  if (pane_rec.x1 * new->scale + new->off_x > bitmap->width)
    new->off_x = bitmap->width - pane_rec.x1 * new->scale;
  if (new->off_x < 0) new->off_x = 0;
  if (pane_rec.y1 * new->scale + new->off_y > bitmap->height)
    new->off_y = bitmap->height - pane_rec.y1 * new->scale;
  if (new->off_y < 0) new->off_y = 0;
  display_xform = *new;

  update_scrollbars(new);

  flow_render_region ();
}

void scroll (int delta_x, int delta_y) {
  xform x;

  x.scale = display_xform.scale;
  x.off_x = display_xform.off_x + delta_x;
  x.off_y = display_xform.off_y + delta_y;
  update_display_xform (&x);
}

void screen_scroll (int delta_x, int delta_y) {
  scroll(display_xform.scale * delta_x, display_xform.scale * delta_y);
}

void scroll_to (double new_x, double new_y) {
  xform x;

  x.scale = display_xform.scale;
  x.off_x = new_x;
  x.off_y = new_y;
  update_display_xform (&x);
}

void update_scale (int new_scale) {
  xform x;
  int old_width, new_width;
  int old_height, new_height;

  x = display_xform;

  old_width = pane_rec.x1 * x.scale;
  if (old_width > bitmap->width)
    old_width = bitmap->width;
  old_height = pane_rec.y1 * x.scale;
  if (old_height > bitmap->height)
    old_height = bitmap->height;

  new_width = pane_rec.x1 * new_scale;
  if (new_width > bitmap->width)
    new_width = bitmap->width;
  new_height = pane_rec.y1 * new_scale;
  if (new_height > bitmap->height)
    new_height = bitmap->height;
  
  x.scale = new_scale;
  x.off_x += (old_width - new_width) / 2;
  x.off_y += (old_height - new_height) / 2;
  
  update_display_xform (&x);
}

void init_scale() {
    update_scale (display_xform.scale);
}

void increment_scale() {
  if (display_xform.scale < 15) update_scale (display_xform.scale + 1);
}

void decrement_scale() {
  if (display_xform.scale > 1) update_scale (display_xform.scale - 1);
}

/* Generate the rectangle that fully encloses the disk. */
void rect_from_disk (rect *dst, const point *ctr, double size) {
  dst->x0 = floor (ctr->x + 1 - size);
  dst->x1 = ceil (ctr->x + size);
  dst->y0 = floor (ctr->y + 1 - size);
  dst->y1 = ceil (ctr->y + size);
}

static Blob *
pen_ellipse(input_state *ins, ToolInfo *tool_info)
{
  double size;
  point p_scr, p_bit;
  double tsin, tcos;
  double aspect, radmin;
  double x,y;
  
  size = tool_info->size * (1 + tool_info->sens * (2*ins->pressure - 1));
  if (size < 1) size = 1;

  p_scr.x = ins->x;
  p_scr.y = ins->y;
  xform_point (&p_bit, &display_xform, &p_scr);

  /* Add brush angle/aspect to title vectorially */

  x = tool_info->aspect*cos(tool_info->angle) + ins->xtilt*10.0;
  y = tool_info->aspect*sin(tool_info->angle) + ins->ytilt*10.0;
  aspect = sqrt(x*x+y*y);

  if (aspect != 0)
    {
      tcos = x/aspect;
      tsin = y/aspect;
    }
  else
    {
      tsin = sin(tool_info->angle);
      tcos = cos(tool_info->angle);
    }
  
  if (aspect < 1.0) 
    aspect = 1.0;
  else if (aspect > 10.0) 
    aspect = 10.0;

  radmin = size/aspect;
  if (radmin < 1.0) radmin = 1.0;
  
  return blob_ellipse(p_bit.x, p_bit.y,
		      radmin*aspect*tcos, radmin*aspect*tsin,  
		      -radmin*tsin, radmin*tcos);
}

/* Draw a disk. */
void draw_begin (input_state *ins, ToolInfo *tool_info) {
  rect rec;
  Blob *b;

  undo_begin ();

  b = pen_ellipse (ins, tool_info);

  blob_bounds (b, &rec);
  bitmap_update (&rec);
  undo_update (&rec);
  bitmap_draw_blob (b,(tool_info->tool == PEN_TOOL) ? 0 : 1);
  free (b);
}

/* Draw a line connecting the last drawn disk to a new disk. */
void draw_update (input_state *last, input_state *ins, ToolInfo *tool_info)
{
  Blob *b0, *b1,*b_union;
  rect rec;

  b0 = pen_ellipse (last, tool_info);
  b1 = pen_ellipse (ins, tool_info);
  b_union = blob_convex_union (b0, b1);
  free(b0);
  free(b1);

  blob_bounds (b_union, &rec);
  bitmap_update (&rec);
  undo_update (&rec);

  bitmap_draw_blob (b_union, (tool_info->tool == PEN_TOOL) ? 0 : 1);
  free (b_union);
}

void draw_end (input_state *ins, ToolInfo *tool_info)
{
  undo_end ();
}

/* Draw a chunk of the draw_region. */
int draw (GtkWidget *area) {
  rect rec;

  if (region_empty (draw_region2)) 
    return 0;
  region_tile (&rec, draw_region2, image_width, 32, 1000);
#ifdef DEBUG
  printf ("drawing (%g, %g) - (%g, %g)\n",
	  rec.x0, rec.y0, rec.x1, rec.y1);
#endif
  gdk_draw_image(area->window,
		 area->style->fg_gc[GTK_WIDGET_STATE (area)],
		 img,
		 rec.x0, rec.y0,
		 rec.x0, rec.y0,
		 rec.x1 - rec.x0, rec.y1 - rec.y0);
  draw_region2 = region_minus (draw_region2, region_from_rect (&rec));
  flow_draw_region ();

  return !region_empty (draw_region2);
}

int iabs (int x) {
  return x >= 0 ? x : -x;
}

/* Initialize the rendering state. */
void render_init (void) {
  rect empty = {0, 0, 0, 0};

  display_xform.scale = 8;
  display_xform.off_x = 0;
  display_xform.off_y = 0;

  display_xform2 = display_xform;

  render_region = region_from_rect (&empty);
  render_region2 = region_from_rect (&empty);
  draw_region = region_from_rect (&empty);
  draw_region2 = region_from_rect (&empty);
}
