package App::Zealc::Command::download;

use 5.014000;
use strict;
use warnings;

our $VERSION = '0.000_001';

use App::Zealc '-command';

use File::Spec::Functions qw/rel2abs/;

use Term::ANSIColor qw/:constants/;
use Term::FormatColumns qw/format_columns/;
use Zeal::Feed;

use constant FEEDS => [qw/NET_Framework ActionScript ColdFusion Akka Android AngularJS Ansible Apache_HTTP_Server Appcelerator_Titanium AppleScript Arduino AWS_JavaScript BackboneJS Bash Boost Bootstrap_2 Bootstrap_3 Bourbon C CakePHP Cappuccino Chai Chef Clojure CMake Cocos2D Cocos3D CodeIgniter CoffeeScript Common_Lisp Compass Cordova Corona CSS D3JS Dart Django Dojo DOM Drupal_7 Drupal_8 ElasticSearch Elixir Emacs_Lisp EmberJS Emmet Erlang Express ExpressionEngine ExtJS Flask Font_Awesome Foundation Git GLib Go Grails Groovy Groovy_JDK Grunt Haml Haskell HTML iOS Jade Jasmine Java_EE6 Java_EE7 Java_SE6 Java_SE7 Java_SE8 JavaFX JavaScript Joomla jQuery jQuery_Mobile jQuery_UI KnockoutJS Kobold2D LaTeX Laravel Less Linux_Man_Pages Mac_OS_X Man_Pages MarionetteJS Markdown Meteor MomentJS MongoDB Mongoose Mono MooTools MySQL Neat Nginx NodeJS NumPy OCaml OpenCV_C OpenCV_Java OpenCV_Python OpenGL_2 OpenGL_3 OpenGL_4 Perl PhoneGap PHP PHPUnit Play_Java Play_Scala PostgreSQL Processing PrototypeJS Puppet Python_2 Python_3 Qt_4 Qt_5 R Redis RequireJS Ruby Ruby_2 Ruby_on_Rails_3 Ruby_on_Rails_4 RubyMotion Rust SaltStack Sass Scala SciPy Sencha_Touch Sinon Smarty Sparrow Spring_Framework SproutCore SQLAlchemy SQLite Statamic Stylus SVG Swift Symfony Tcl Tornado Twig Twisted TYPO3 UnderscoreJS Unity_3D Vagrant Vim VMware_vSphere WordPress Xamarin Xojo XSLT XUL Yii YUI Zend_Framework_1 Zend_Framework_2 ZeptoJS/];

my %feeds = map { y/A-Z_/a-z-/r => $_ } @{FEEDS()};

sub usage_desc { "%c download %o\n%c download %o DOCSET\n%c download URL_TO_FEED" }

sub validate_args {
	my ($self, $opts, $args) = @_;
	my @args = @$args;
	$self->usage_error("Too many arguments") if @args > 1;
}

sub _show_docset {
	my ($file) = @_;
	return $_                   unless -d $file;
	return BOLD . "$_*" . RESET if -t select;
	return "$_*";
}

sub execute {
	my ($self, $opts, $args) = @_;
	my ($name) = @$args;
	if (!$name) {
		my @feeds = map {
			my $fname = $feeds{$_} =~ y/_/ /r;
			my $file = rel2abs "$fname.docset", $self->app->path;
			_show_docset $file
		} sort keys %feeds;
		print format_columns @feeds;
	} else {
		my $feed = Zeal::Feed->new(exists $feeds{$name} ? "http://kapeli.com/feeds/$feeds{$name}.xml" : $name);
		say "Downloading $feeds{$name} version ", $feed->version, ' from ', $feed->url;
		$feed->download($self->app->path);
	}
}

1;
__END__

=encoding utf-8

=head1 NAME

App::Zealc::Command::download - view available docsets and download them

=head1 SYNOPSIS

  zealc download
  # <list of docsets>

  zealc download latex
  # Downloading LaTeX version /1 from http://london3.kapeli.com/feeds/LaTeX.tgz

  zealc download http://kapeli.com/feeds/LaTeX.xml
  # Downloading LaTeX version /1 from http://london3.kapeli.com/feeds/LaTeX.tgz

=head1 DESCRIPTION

When invoked with no arguments, the download command displays a list
of known Dash docsets. The installed docsets have a * after their
name.

When invoked with an argument from the list of known Dash docsets it
downloads that docset.

When invoked with an argument not from the list of known Dash docsets
it assumes the argument is a link to a L<Zeal::Feed> and downloads the
docset described by the feed.

=head1 SEE ALSO

L<zealc>, L<Zeal>

=head1 AUTHOR

Marius Gavrilescu, E<lt>marius@ieval.roE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2015 by Marius Gavrilescu

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.20.1 or,
at your option, any later version of Perl 5 you may have available.


=cut
